package jp.veltec.pdf;
import java.io.OutputStream;
import java.io.IOException;
import java.io.File;
import java.util.Map;
import java.util.HashMap;
import java.net.MalformedURLException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.NodeList;

import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.pdf.PdfWriter;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfImportedPage;
import com.itextpdf.text.Font;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.BadElementException;
import com.itextpdf.text.Image;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.ExceptionConverter;

/**
*1ページのレイアウトを行うための基底クラスです。
*/
public abstract class PdfLayout{

	OutputStream fOutputStream;
	org.w3c.dom.Element fTemplate;
	Document fDocument;
	PdfWriter fWriter;
	PageNumberWriter fPageNumberWriter;
	PageHeaderWriter fPageHeaderWriter;
	PdfContentByte fDirectContent;
	PdfContentByte fDirectContentUnder;
	String fImagePath=".";
	HashMap<String,HashMap<String,String>> fExtraAttributes =  new HashMap<String,HashMap<String,String>>();
	HashMap<String,String> fData =  new HashMap<String,String>();

	public PdfLayout(OutputStream aOutputStream){
		try{
			fOutputStream = aOutputStream;
	        fDocument = new Document();
	        fWriter = PdfWriter.getInstance(fDocument, fOutputStream);
		}catch(DocumentException e){
			e.printStackTrace();
		}
	}

	/**
	* テンプレートに埋め込むデータを指定します。
	* @param aKey 値を埋め込む要素のID
	* @param aValue 埋め込む値
	*/
	public void setData(String aKey,String aValue){
		fData.put(aKey,aValue);
	}

	/**
	* 埋め込みデータをクリアします。
	*/
	public void clearData(){
		fData.clear();
	}

	/**
	* 属性をセットします。
	* @param aExtraAttributes セット先の要素をキーに属性の[名前=値]ペアを保持するHashMap
	*/
	public void setAttributes(HashMap<String,HashMap<String,String>> aExtraAttributes){
		fExtraAttributes = aExtraAttributes;
	}

	/**
	* setAttributesでセットした属性をクリアします。
	*/
	public void clearAttributes(){
		fExtraAttributes.clear();
	}

	/**
	* 属性をセットします。
	* @param aId 属性をセットする要素のID
	* @param aKey 属性名
	* @param aValue 属性値
	*/
	public void setAttribute(String aId, String aKey,String aValue){
		HashMap<String,String> attributes = fExtraAttributes.get(aId);
		if(attributes==null){
			attributes=new HashMap<String,String>();
			fExtraAttributes.put(aId,attributes);
		}
		attributes.put(aKey,aValue);
	}

	/**
	* 属性をクリアします。
	* @param aId 属性をセットする要素のID
	* @param aKey 属性名
	*/
	public void removeAttribute(String aId,String aKey){
		HashMap attributes = fExtraAttributes.get(aId);
		if(attributes==null){
			throw new IllegalArgumentException(aId + " not exists.");
		}
		attributes.remove(aKey);
	}

	/**
	* 画像ファイルのデフォルトパスをセットします。
	* @param aImagePath 画像ファイルのデフォルトパス
	*/
	public void setImagePath(String aImagePath){
		fImagePath = aImagePath;
	}

	/**
	* フォントファイルのデフォルトパスをセットします。
	* @param aDefaultFontPath フォントファイルのデフォルトパス
	*/
	public void setFontPath(String aDefaultFontPath){
		FontManager.setDefaultFontPath(aDefaultFontPath);
	}

	/**
	* ページ番号の初期値をセットします。
	* @param aPageNumber ページ番号
	*/
	public void setPageNumber(int aPageNumber){
		fPageNumberWriter.setPageNumber(aPageNumber);
	}

	/**
	* ページヘッダーのテキストをセットします。
	* @param aHeaderText ヘッダーのテキスト
	*/
	public void setPageHeaderText(String aHeaderText){
		fPageHeaderWriter.setHeaderText(aHeaderText);
	}

	/**
	* テンプレートファイルをセットします。
	* @param aFile テンプレートファイル
	*/
	public void setTemplateFile(File aFile)throws IOException{
		try{
			DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			org.w3c.dom.Document xmlDocument = builder.parse(aFile);
			org.w3c.dom.Element xmlElement = xmlDocument.getDocumentElement();
			setTemplate(xmlElement);
		}catch(org.xml.sax.SAXException e){
			e.printStackTrace();
			throw new IllegalStateException(e.getMessage());
		}catch(javax.xml.parsers.ParserConfigurationException e){
			e.printStackTrace();
			throw new IllegalStateException(e.getMessage());
		}
	}

	/**
	* テンプレートをセットします。
	* @param aTemplate テンプレート
	*/
	public void setTemplate(org.w3c.dom.Element aTemplate){
		fTemplate=aTemplate;
		Rectangle mSize = PageSize.A4;
		String mAttribute = fTemplate.getAttribute("size");
		if(!mAttribute.equals("")){
			String[] size = mAttribute.split(" ");
			if(size.length==1){
				mSize = Util.getStandardSize(mAttribute);
			}else if(size.length==2){
				float x = Util.getSize(size[0]);
				float y = Util.getSize(size[1]);
				mSize = new Rectangle(x,y);
			}else{
				throw new IllegalArgumentException("not supported document size:" + mAttribute);
			}
		}
		fDocument.setPageSize(mSize);

		float mMarginLeft=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-left");
		if(!mAttribute.equals("")){
			mMarginLeft=Util.getSize(mAttribute);
		}

		float mMarginRight=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-right");
		if(!mAttribute.equals("")){
			mMarginRight=Util.getSize(mAttribute);
		}

		float mMarginTop=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-top");
		if(!mAttribute.equals("")){
			mMarginTop=Util.getSize(mAttribute);
		}

		float mMarginBottom=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-bottom");
		if(!mAttribute.equals("")){
			mMarginBottom=Util.getSize(mAttribute);
		}

		fDocument.setMargins(mMarginLeft,mMarginRight,mMarginTop,mMarginBottom);

		NodeList listFont = fTemplate.getElementsByTagName("font");
		if(listFont.getLength()!=0){
			FontManager.clearFonts();
			FontManager.register(listFont);
		}

		NodeList pageNumbers = fTemplate.getElementsByTagName("page");
		if(pageNumbers.getLength()!=0){
			if(pageNumbers.getLength()!=1){
				throw new IllegalArgumentException("more than one page elements found");
			}
			fPageNumberWriter= new PageNumberWriter(this);
			fPageNumberWriter.setDocument(fDocument);
			fPageNumberWriter.setPageLabel((org.w3c.dom.Element)pageNumbers.item(0));
			fWriter.setPageEvent(fPageNumberWriter);
		}

		NodeList pageHeaders = fTemplate.getElementsByTagName("page-header");
		if(pageHeaders.getLength()!=0){
			if(pageHeaders.getLength()!=1){
				throw new IllegalArgumentException("more than one page-header elements found");
			}
			fPageHeaderWriter= new PageHeaderWriter(this);
			fPageHeaderWriter.setDocument(fDocument);
			fPageHeaderWriter.setPageLabel((org.w3c.dom.Element)pageHeaders.item(0));
			fWriter.setPageEvent(fPageHeaderWriter);
		}

        fDocument.open();
		fDirectContent = fWriter.getDirectContent();
		fDirectContentUnder = fWriter.getDirectContentUnder();
	}

	/**
	* 属性とデータを確定し、ページを追加します。
	*/
    public void addPage() throws IllegalStateException{
		addPage(fData);
	}

	/**
	* 埋め込む値を指定してページを追加します。
	* @param aData 埋め込む値を[ID=値]形式でセットしたHashMap
	*/
    public void addPage(HashMap<String,String> aData) throws IllegalStateException{
		try{
			process(aData);
			processPDF();
			processImage();
			processRectangle();
			processLine();
			fDocument.newPage();
		}catch(MalformedURLException e){
			throw new IllegalStateException(e.getMessage());
		}catch(IOException e){
			throw new IllegalStateException(e.getMessage());
		}catch(DocumentException e){
			throw new IllegalStateException(e.getMessage());
		}
    }

    abstract void process(HashMap<String,String> aData)throws DocumentException;

	/**
	* 帳票をクローズします。
	*/
	public void close() throws PdfException{
		try{
	        fDocument.close();
		}catch(ExceptionConverter e){
			throw new PdfException(e.getMessage());
		}
	}

	void processImage()throws BadElementException,DocumentException,MalformedURLException,IOException{
		NodeList images = fTemplate.getElementsByTagName("img");
		for(int i=0;i<images.getLength();i++){
			org.w3c.dom.Element item = (org.w3c.dom.Element)images.item(i);
			String mId = item.getAttribute("id");
			if(!mId.equals("")){
				HashMap<String,String> mExtraAttributes = fExtraAttributes.get(mId);
				if(mExtraAttributes!=null){
					for(Map.Entry<String,String> entry:mExtraAttributes.entrySet()){
						item.setAttribute(entry.getKey(),entry.getValue());
					}
				}
			}
			String mFileName = item.getAttribute("src");
			String mFilePath = item.getAttribute("path");
			if(!mFilePath.equals("")){
				mFileName = mFilePath + "/" + mFileName;
			}else{
				mFileName = fImagePath + "/" + mFileName;
			}
			Image image = Image.getInstance(mFileName);
			String mAttribute = null;
			String mWidth = item.getAttribute("width");
			String mHeight = item.getAttribute("height");
			float width = image.getWidth();
			float height = image.getHeight();
			if(!mWidth.equals("")){
				width=Util.getSize(mWidth);
				if(mHeight.equals("")){
					height=image.getHeight()*width/image.getWidth();
				}
			}
			if(!mHeight.equals("")){
				height=Util.getSize(mHeight);
				if(mWidth.equals("")){
					width=image.getWidth()*height/image.getHeight();
				}
			}
			image.scaleAbsolute(width,height);
			float top = Util.getSize(item.getAttribute("top"));
			float left = Util.getSize(item.getAttribute("left"));
			image.setAbsolutePosition(left,fDocument.getPageSize().getHeight()-top-height);
			String isBackGround = item.getAttribute("background");
			boolean mBackground = isBackGround.equals("true");
			if(mBackground){
				fDirectContentUnder.addImage(image);
			}else{
				fDirectContent.addImage(image);
			}
		}
	}

	void processPDF()throws BadElementException,DocumentException,MalformedURLException,IOException{
		try{
			NodeList items = fTemplate.getElementsByTagName("pdf");
			if(items.getLength()>1){
				throw new IllegalArgumentException("only one pdf can be included.");
			}
			org.w3c.dom.Element item = (org.w3c.dom.Element)items.item(0);
			if(item==null){
				return;
			}
			String mId = item.getAttribute("id");
			if(!mId.equals("")){
				HashMap<String,String> mExtraAttributes = fExtraAttributes.get(mId);
				if(mExtraAttributes!=null){
					for(Map.Entry<String,String> entry:mExtraAttributes.entrySet()){
						item.setAttribute(entry.getKey(),entry.getValue());
					}
				}
			}
			String mFileName = item.getAttribute("src");
			String mFilePath = item.getAttribute("path");
			if(!mFilePath.equals("")){
				mFileName = mFilePath + "/" + mFileName;
			}else{
				mFileName = fImagePath + "/" + mFileName;
			}
			String mAttribute = item.getAttribute("page");
			int mPage = 1;
			if(!mAttribute.equals("")){
				mPage=Integer.parseInt(mAttribute);
			}
			mAttribute = item.getAttribute("left");
			float mLeft = 0;
			if(!mAttribute.equals("")){
				mLeft=Util.getSize(mAttribute);
			}
			mAttribute = item.getAttribute("top");
			float mTop = 0;
			if(!mAttribute.equals("")){
				mTop=Util.getSize(mAttribute);
			}
			PdfReader mReader = new PdfReader(mFileName);
			PdfImportedPage mPdfPage = fWriter.getImportedPage(mReader,mPage);
			Image mImage = Image.getInstance(mPdfPage);
			mImage.setAbsolutePosition(mLeft,fDocument.getPageSize().getHeight()-mImage.getHeight()-mTop);
			String isBackGround = item.getAttribute("background");
			boolean mBackground = isBackGround.equals("true");
			if(mBackground){
				fDirectContentUnder.addImage(mImage);
			}else{
				fDirectContent.addImage(mImage);
			}

		}catch(Exception e){
			if(e.getMessage().indexOf("password")!=-1){
				throw new IllegalArgumentException("PDF requires password, not supported.");
			}
			throw new IllegalArgumentException(e.getMessage());
		}
		
	}

	void processRectangle()throws BadElementException,DocumentException,MalformedURLException,IOException{
		NodeList images = fTemplate.getElementsByTagName("rect");
		for(int i=0;i<images.getLength();i++){
			org.w3c.dom.Element item = (org.w3c.dom.Element)images.item(i);
			String mId = item.getAttribute("id");
			if(!mId.equals("")){
				HashMap<String,String> mExtraAttributes = fExtraAttributes.get(mId);
				if(mExtraAttributes!=null){
					for(Map.Entry<String,String> entry:mExtraAttributes.entrySet()){
						item.setAttribute(entry.getKey(),entry.getValue());
					}
				}
			}
			PdfContentByte mDirectContent=null;
			if(item.getAttribute("background").equals("true")){
				mDirectContent = fDirectContentUnder;
			}else{
				mDirectContent = fDirectContent;
			}
			mDirectContent.saveState();
			float mLeft = Util.getSize(item.getAttribute("left"));
			float mWidth = Util.getSize(item.getAttribute("width"));
			float mHeight = Util.getSize(item.getAttribute("height"));
			float mTop = fDocument.getPageSize().getHeight()-Util.getSize(item.getAttribute("top"))-mHeight;
			String attribute = item.getAttribute("border-width");
			if(!attribute.equals("")){
				mDirectContent.setLineWidth(Util.getSize(attribute));
			}
			attribute = item.getAttribute("border-color");
			if(!attribute.equals("")){
				mDirectContent.setColorStroke(Util.getColor(attribute));
			}
			mDirectContent.setColorFill(BaseColor.WHITE);
			attribute = item.getAttribute("color");
			if(!attribute.equals("")){
				mDirectContent.setColorFill(Util.getColor(attribute));
			}
			attribute = item.getAttribute("round");
			if(!attribute.equals("")){
				float mRound = Util.getSize(attribute);
				mDirectContent.roundRectangle(mLeft,mTop,mWidth,mHeight,mRound);
			}else{
				mDirectContent.rectangle(mLeft,mTop,mWidth,mHeight);
			}
			mDirectContent.fillStroke();
			mDirectContent.restoreState();
		}
	}

	void processLine()throws BadElementException,DocumentException,MalformedURLException,IOException{
		NodeList images = fTemplate.getElementsByTagName("line");
		for(int i=0;i<images.getLength();i++){
			org.w3c.dom.Element item = (org.w3c.dom.Element)images.item(i);
			String mId = item.getAttribute("id");
			if(!mId.equals("")){
				HashMap<String,String> mExtraAttributes = fExtraAttributes.get(mId);
				if(mExtraAttributes!=null){
					for(Map.Entry<String,String> entry:mExtraAttributes.entrySet()){
						item.setAttribute(entry.getKey(),entry.getValue());
					}
				}
			}
			PdfContentByte mDirectContent=null;
			if(item.getAttribute("background").equals("true")){
				mDirectContent = fDirectContentUnder;
			}else{
				mDirectContent = fDirectContent;
			}
			mDirectContent.saveState();
			float mX0 = Util.getSize(item.getAttribute("from").split(" ")[0]);
			float mY0 = fDocument.getPageSize().getHeight()-Util.getSize(item.getAttribute("from").split(" ")[1]);
			float mX1 = Util.getSize(item.getAttribute("to").split(" ")[0]);
			float mY1 = fDocument.getPageSize().getHeight()-Util.getSize(item.getAttribute("to").split(" ")[1]);
			String attribute = item.getAttribute("width");
			if(!attribute.equals("")){
				mDirectContent.setLineWidth(Util.getSize(attribute));
			}
			attribute = item.getAttribute("color");
			if(!attribute.equals("")){
				mDirectContent.setColorStroke(Util.getColor(attribute));
			}
			mDirectContent.moveTo(mX0,mY0);
			mDirectContent.lineTo(mX1,mY1);
			mDirectContent.stroke();
			mDirectContent.restoreState();
		}
	}

}
