package jp.veltec.pdf;

import java.util.Map;
import java.util.HashMap;
import java.util.ArrayList;
import java.io.OutputStream;
import java.io.IOException;
import java.io.File;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.NodeList;
import org.w3c.dom.Node;
import org.w3c.dom.NamedNodeMap;

import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.pdf.PdfWriter;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.Font;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Element;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Image;
import com.itextpdf.text.pdf.ColumnText;
import com.itextpdf.text.pdf.PdfPageEventHelper;
import com.itextpdf.text.ExceptionConverter;

import com.itextpdf.text.BadElementException;
import java.net.MalformedURLException;

/**
 *複数ページにわたる表を含む帳票を作成します。
 */
public class ListTableLayout{

	OutputStream fOutputStream;

	org.w3c.dom.Element fTemplate;
    Document fDocument;
	float fDocumentHeight;

	PdfWriter fWriter;
	HashMap<String,String> fTableDefault;
	HashMap<String,String> fRowDefault;
	HashMap<String,String> fHeaderData = new HashMap<String,String>();
	HashMap<String,String> fRecordData = new HashMap<String,String>();
	HashMap<String,String> fFooterData = new HashMap<String,String>();
	HashMap<String,HashMap<String,String>> fExtraAttributes = new HashMap<String,HashMap<String,String>>();

	NodeList fHeaderTrs;
	NodeList fRecordTrs;
	NodeList fFooterTrs;

	PdfPTable fPdfTable;
	int fRows;
	int fRowCounter;
	int fCols;
	int fHeaderRows;
	boolean[][] fMatrix;
	boolean fRepeat;
	String fName;
	PageHeaderWriter fPageHeaderWriter;

	public ListTableLayout(OutputStream aOutputStream){
		fOutputStream = aOutputStream;
		try{
			fDocument = new Document();
	        fWriter = PdfWriter.getInstance(fDocument, fOutputStream);
		}catch(DocumentException e){
			e.printStackTrace();
			throw new IllegalStateException(e.getMessage());
		}
	}

	/**
	* ページヘッダーのテキストをセットします。
	* @param aHeaderText ヘッダーのテキスト
	*/
	public void setPageHeaderText(String aHeaderText){
		fPageHeaderWriter.setHeaderText(aHeaderText);
	}

	/**
	* ヘッダー行にデータをセットします。
	* @param aKey 値を埋め込む要素のID
	* @param aValue 埋め込む値
	*/
	public void setHeaderData(String aKey,String aValue){
		fHeaderData.put(aKey,aValue);
	}

	/**
	* レコード行にデータをセットします。
	* @param aKey 値を埋め込む要素のID
	* @param aValue 埋め込む値
	*/
	public void setRecordData(String aKey,String aValue){
		fRecordData.put(aKey,aValue);
	}

	/**
	* フッター行にデータをセットします。
	* @param aKey 値を埋め込む要素のID
	* @param aValue 埋め込む値
	*/
	public void setFooterData(String aKey,String aValue){
		fFooterData.put(aKey,aValue);
	}

	/**
	* ヘッダー行のデータをクリアします。
	*/
	public void clearHeaderData(){
		fHeaderData.clear();
	}

	/**
	* レコード行のデータをクリアします。
	*/
	public void clearRecordData(){
		fRecordData.clear();
	}

	/**
	* フッター行のデータをクリアします。
	*/
	public void clearFooterData(){
		fFooterData.clear();
	}

	/**
	* 属性をセットします。
	* @param aExtraAttributes セット先の要素をキーに属性の[名前=値]ペアを保持するHashMap
	*/
	public void setAttributes(HashMap<String,HashMap<String,String>> aExtraAttributes){
		fExtraAttributes = aExtraAttributes;
	}

	/**
	* setAttributesでセットした属性をクリアします。
	*/
	public void clearAttributes(){
		fExtraAttributes.clear();
	}

	/**
	* 属性をセットします。
	* @param aId 属性をセットする要素のID
	* @param aKey 属性名
	* @param aValue 属性値
	*/
	public void setAttribute(String aId, String aKey,String aValue){
		HashMap<String,String> attributes = fExtraAttributes.get(aId);
		if(attributes==null){
			attributes=new HashMap<String,String>();
			fExtraAttributes.put(aId,attributes);
		}
		attributes.put(aKey,aValue);
	}

	/**
	* 属性をクリアします。
	* @param aId 属性をセットする要素のID
	* @param aKey 属性名
	*/
	public void removeAttribute(String aId,String aKey){
		HashMap attributes = fExtraAttributes.get(aId);
		if(attributes==null){
			throw new IllegalArgumentException(aId + " not exists.");
		}
		attributes.remove(aKey);
	}

	/**
	* フォントファイルのデフォルトパスをセットします。
	* @param aFontPath フォントファイルのデフォルトパス
	*/
	public void setFontPath(String aFontPath){
		FontManager.setDefaultFontPath(aFontPath);
	}

	/**
	* テンプレートファイルをセットします。
	* @param aFile テンプレートファイル
	*/
	public void setTemplateFile(File aFile){
		try{
			DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			org.w3c.dom.Document xmlDocument = builder.parse(aFile);
			org.w3c.dom.Element xmlElement = xmlDocument.getDocumentElement();
			setTemplate(xmlElement);
		}catch(org.xml.sax.SAXException e){
			e.printStackTrace();
			throw new IllegalStateException(e.getMessage());
		}catch(javax.xml.parsers.ParserConfigurationException e){
			e.printStackTrace();
			throw new IllegalStateException(e.getMessage());
		}catch(IOException e){
			e.printStackTrace();
			throw new IllegalStateException(e.getMessage());
		}
	}

	/**
	* テンプレートをセットします。
	* @param aTemplate テンプレート
	*/
	public void setTemplate(org.w3c.dom.Element aTemplate){
		fTemplate=aTemplate;
		Rectangle mSize = PageSize.A4;
		String mAttribute = fTemplate.getAttribute("size");
		if(!mAttribute.equals("")){
			String[] size = mAttribute.split(" ");
			if(size.length==1){
				mSize = Util.getStandardSize(mAttribute);
			}else if(size.length==2){
				float x = Util.getSize(size[0]);
				float y = Util.getSize(size[1]);
				mSize = new Rectangle(x,y);
			}else{
				throw new IllegalArgumentException("not supported document size:" + mAttribute);
			}
		}
		fDocument.setPageSize(mSize);

		float mMarginLeft=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-left");
		if(!mAttribute.equals("")){
			mMarginLeft=Util.getSize(mAttribute);
		}

		float mMarginRight=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-right");
		if(!mAttribute.equals("")){
			mMarginRight=Util.getSize(mAttribute);
		}

		float mMarginTop=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-top");
		if(!mAttribute.equals("")){
			mMarginTop=Util.getSize(mAttribute);
		}

		float mMarginBottom=Util.getSize("15mm");
		mAttribute  = fTemplate.getAttribute("margin-bottom");
		if(!mAttribute.equals("")){
			mMarginBottom=Util.getSize(mAttribute);
		}

		NodeList fonts = fTemplate.getElementsByTagName("font");
		if(fonts.getLength()!=0){
			FontManager.clearFonts();
			FontManager.register(fonts);
		}

		NodeList pageNumbers = fTemplate.getElementsByTagName("page");
		if(pageNumbers.getLength()!=0){
			if(pageNumbers.getLength()!=1){
				throw new IllegalArgumentException("more than one page elements found");
			}
			PageNumberWriter pageNumberWriter= new PageNumberWriter(this);
			pageNumberWriter.setDocument(fDocument);
			pageNumberWriter.setPageLabel((org.w3c.dom.Element)pageNumbers.item(0));
			fWriter.setPageEvent(pageNumberWriter);
		}

		NodeList pageHeaders = fTemplate.getElementsByTagName("page-header");
		if(pageHeaders.getLength()!=0){
			if(pageHeaders.getLength()!=1){
				throw new IllegalArgumentException("more than one page-header elements found");
			}
			fPageHeaderWriter= new PageHeaderWriter(this);
			fPageHeaderWriter.setDocument(fDocument);
			fPageHeaderWriter.setPageLabel((org.w3c.dom.Element)pageHeaders.item(0));
			fWriter.setPageEvent(fPageHeaderWriter);
		}

		fDocument.setMargins(mMarginLeft,mMarginRight,mMarginTop,mMarginBottom);
        fDocument.open();
		beginTable();
	}

    private void beginTable() throws IllegalStateException{
		try{
			NodeList tables = fTemplate.getElementsByTagName("listtable");
			org.w3c.dom.Element xmlTable = (org.w3c.dom.Element)tables.item(0);
			String strCols = xmlTable.getAttribute("cols");
			org.w3c.dom.Element header = (org.w3c.dom.Element)xmlTable.getElementsByTagName("header").item(0);
			if(header!=null){
				fHeaderTrs = header.getElementsByTagName("tr");
				fHeaderRows = fHeaderTrs.getLength();
			}
			org.w3c.dom.Element record = (org.w3c.dom.Element)xmlTable.getElementsByTagName("record").item(0);
			if(record!=null){
				fRecordTrs = record.getElementsByTagName("tr");
			}
			org.w3c.dom.Element footer = (org.w3c.dom.Element)xmlTable.getElementsByTagName("footer").item(0);
			if(footer!=null){
				fFooterTrs = footer.getElementsByTagName("tr");
			}
			if(strCols.equals("")){
				throw new IllegalStateException("cols not specified");
			}
			fCols = Integer.parseInt(strCols);
			fPdfTable = new PdfPTable(fCols);
			fPdfTable.setLockedWidth(true);
			fPdfTable.setTotalWidth(fDocument.getPageSize().getWidth()-fDocument.leftMargin()-fDocument.rightMargin());
			fTableDefault = new HashMap<String,String>();
			NamedNodeMap mapT = xmlTable.getAttributes();
			fRepeat=false;
			for(int i=0;i<mapT.getLength();i++){
				Node node = mapT.item(i);
				if(node.getNodeName().equals("widths")){
					String[] strWidths = node.getNodeValue().split(" ");
					float[] widths = new float[strWidths.length];
					float totalWidth=0;
					for(int j=0;j<widths.length;j++){
						widths[j]=Util.getSize(strWidths[j]);
						totalWidth+=widths[j];
					}
					fPdfTable.setWidths(widths);
					fPdfTable.setTotalWidth(totalWidth);
				}else if(node.getNodeName().equals("name")){
					fName = node.getNodeValue();
				}else if(node.getNodeName().equals("outer-border-width")){
					DrawOuterBorder outerBorder = new DrawOuterBorder();
					outerBorder.setBorderWidth(Util.getSize(node.getNodeValue()));
					fPdfTable.setTableEvent(outerBorder);
				}else if(node.getNodeName().equals("table-align")){
					String attribute = node.getNodeValue();
					if(attribute.equals("left")){
						fPdfTable.setHorizontalAlignment(0);
					}else if(attribute.equals("center")){
						fPdfTable.setHorizontalAlignment(1);
					}else if(attribute.equals("right")){
						fPdfTable.setHorizontalAlignment(2);
					}else{
						throw new IllegalArgumentException("bad table-align:" + attribute);
					}
				}else{
					fTableDefault.put(node.getNodeName(),node.getNodeValue());
				}
			}
		}catch(DocumentException e){
			throw new IllegalStateException(e.getMessage());
		}
    }

	/**
	* ヘッダー行を追加します。
	*/
	public void addHeader() throws IllegalStateException{
		addHeader(fHeaderData);
	}

	/**
	* 埋め込む値をセットしてヘッダー行を追加します。
	* @param aData 埋め込む値をセットしたHashMap
	*/
	public void addHeader(HashMap<String,String> aData) throws IllegalStateException{
		try{
			ArrayList<String> mIdList = new ArrayList<String>();
			fPdfTable.setHeaderRows(fHeaderRows);
			if(fHeaderTrs==null){
				throw new IllegalStateException("header is not found in the template");
			}
			for(int i=0;i<fHeaderTrs.getLength();i++){
				org.w3c.dom.Element tr = (org.w3c.dom.Element)fHeaderTrs.item(i);
				fRowDefault = new HashMap<String,String>();
				NamedNodeMap mapR = tr.getAttributes();
				for(int j=0;j<mapR.getLength();j++){
					Node nodeR = mapR.item(j);
					fRowDefault.put(nodeR.getNodeName(),nodeR.getNodeValue());
				}
				processRow(tr,aData,mIdList);
			}
			for(String key:aData.keySet()){
				if(!mIdList.contains(key)){
					throw new IllegalArgumentException("id not found:" + key);
				}
			}
		}catch(DocumentException e){
			throw new IllegalStateException(e.getMessage());
		}
	}

	/**
	* レコード行を追加します。
	*/
	public void addRecord() throws IllegalStateException{
		addRecord(fRecordData);
	}

	/**
	* 埋め込む値をセットしてレコード行を追加します。
	* @param aData 埋め込む値をセットしたHashMap
	*/
	public void addRecord(HashMap<String,String> aData) throws IllegalStateException{
		try{
			ArrayList<String> mIdList = new ArrayList<String>();
			if(fRecordTrs==null){
				throw new IllegalStateException("record is not found in the template");
			}
			for(int i=0;i<fRecordTrs.getLength();i++){
				org.w3c.dom.Element tr = (org.w3c.dom.Element)fRecordTrs.item(i);
				fRowDefault = new HashMap<String,String>();
				NamedNodeMap mapR = tr.getAttributes();
				for(int j=0;j<mapR.getLength();j++){
					Node nodeR = mapR.item(j);
					fRowDefault.put(nodeR.getNodeName(),nodeR.getNodeValue());
				}
				processRow(tr,aData,mIdList);
			}

			for(String key:aData.keySet()){
				if(!mIdList.contains(key)){
					throw new IllegalArgumentException("id not found:" + key);
				}
			}
			clearAttributes();
		}catch(DocumentException e){
			throw new IllegalStateException(e.getMessage());
		}
	}

	/**
	* フッター行を追加します。
	*/
	public void addFooter() throws IllegalStateException{
		addFooter(fFooterData);
	}

	/**
	* 埋め込む値をセットしてフッター行を追加します。
	* @param aData 埋め込む値をセットしたHashMap
	*/
	public void addFooter(HashMap<String,String> aData) throws IllegalStateException{
		try{
			ArrayList<String> mIdList = new ArrayList<String>();
			if(fFooterTrs==null){
				throw new IllegalStateException("footer is not found in the template");
			}
			for(int i=0;i<fFooterTrs.getLength();i++){
				org.w3c.dom.Element tr = (org.w3c.dom.Element)fFooterTrs.item(i);
				fRowDefault = new HashMap<String,String>();
				NamedNodeMap mapR = tr.getAttributes();
				for(int j=0;j<mapR.getLength();j++){
					Node nodeR = mapR.item(j);
					fRowDefault.put(nodeR.getNodeName(),nodeR.getNodeValue());
				}
				processRow(tr,aData,mIdList);
			}

			for(String key:aData.keySet()){
				if(!mIdList.contains(key)){
					throw new IllegalArgumentException("id not found:" + key);
				}
			}
		}catch(DocumentException e){
			throw new IllegalStateException(e.getMessage());
		}
	}
	

	/**
	* 改ページします。
	*/
	public void newPage() throws IllegalStateException{
		try{
			fDocument.add(fPdfTable);
			fDocument.newPage();
			beginTable();
		}catch(DocumentException e){
			throw new IllegalStateException(e.getMessage());
		}
	}

	/**
	* 帳票をクローズします。
	*/
	public void close() throws PdfException{
		try{
			fDocument.add(fPdfTable);
	        fDocument.close();
		}catch(DocumentException e){
			throw new PdfException(e.getMessage());
		}catch(ExceptionConverter e){
			throw new PdfException(e.getMessage());
		}
	}

	void processRow(org.w3c.dom.Element fTr,HashMap<String,String> aData,ArrayList<String> aIdList) throws DocumentException{
		NodeList tds = fTr.getElementsByTagName("td");
		boolean isLeftEnd;
		boolean isRightEnd;
		boolean isBottomEnd;
		for(int k=0;k<tds.getLength();k++){
			org.w3c.dom.Element td = (org.w3c.dom.Element)tds.item(k).cloneNode(true);
			String mContent = td.getTextContent();
			String mId = td.getAttribute("id");
			if(!mId.equals("")){
				aIdList.add(mId);
				String mValue = aData.get(mId);
				if(mValue!=null){
					mContent = mValue;
				}
				HashMap<String,String> mExtraAttributes = fExtraAttributes.get(mId);
				if(mExtraAttributes!=null){
					for(Map.Entry<String,String> entry:mExtraAttributes.entrySet()){
						td.setAttribute(entry.getKey(),entry.getValue());
					}
				}
			}
			Cell cell = new Cell();
			cell.setTableDefault(fTableDefault);
			cell.setRowDefault(fRowDefault);
			String strColspan = td.getAttribute("colspan");
			int colspan = 1;
			if(!strColspan.equals("")){
				colspan = Integer.parseInt(strColspan);
				cell.setColspan(colspan);
			}
			int colCounter=0;
			int rowspan = 1;
			String strRowspan = td.getAttribute("rowspan");
			if(!strRowspan.equals("")){
				rowspan = Integer.parseInt(strRowspan);
				cell.setRowspan(rowspan);
			}
			cell.setAttributes(td);
			cell.setContent(mContent);
			fPdfTable.addCell(cell);
		}
	}


}
